<?php
function ensure_dirs() {
  if (!is_dir(DATA_DIR)) mkdir(DATA_DIR, 0775, true);
}

function read_json($path, $fallback = []) {
  if (!file_exists($path)) return $fallback;
  $json = file_get_contents($path);
  $data = json_decode($json, true);
  return is_array($data) ? $data : $fallback;
}

function write_json($path, $data) {
  file_put_contents($path, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT));
}

function fetch_autotrader_listings() {
  $items = [];
  $page  = 1;
  while (true) {
    $url = AT_BASE_URL . '?pageSize=' . AT_PAGE_SIZE . '&pageNumber=' . $page;
    $opts = [
      'http' => [
        'method'  => 'GET',
        'timeout' => 90,
        'header'  => "Authorization: Basic " . base64_encode(AT_USERNAME . ':' . AT_PASSWORD) . "\r\n" .
                    "Accept: application/json\r\n",
      ]
    ];
    $ctx = stream_context_create($opts);
    $raw = file_get_contents($url, false, $ctx);
    if ($raw === false || $raw === '') break;
    $json = json_decode($raw, true);
    if (!is_array($json)) break;
    if (isset($json['items'])) {
      $items = array_merge($items, $json['items']);
      $totalPages = isset($json['totalPages']) ? (int)$json['totalPages'] : $page;
      if ($page >= $totalPages) break;
      $page++;
      continue;
    } elseif (isset($json['data'])) {
      $items = array_merge($items, $json['data']);
      if (isset($json['links']['next']) && $json['links']['next']) {
        $nextUrl = $json['links']['next'];
        while ($nextUrl) {
          $raw2 = file_get_contents($nextUrl, false, $ctx);
          if ($raw2 === false || $raw2 === '') { $nextUrl = null; break; }
          $js2  = json_decode($raw2, true);
          $b2   = isset($js2['data']) ? $js2['data'] : [];
          $items = array_merge($items, $b2);
          $nextUrl = isset($js2['links']['next']) ? $js2['links']['next'] : null;
        }
      }
      break;
    } elseif (array_is_list($json)) {
      $items = array_merge($items, $json);
      break;
    } else {
      break;
    }
  }
  return $items;
}

function normalise_vehicle($v) {
  $id   = $v['id'] ?? $v['listingId'] ?? $v['stockId'] ?? null;
  $yr   = $v['year'] ?? $v['modelYear'] ?? null;
  $mk   = $v['make'] ?? ($v['vehicle']["make"] ?? null);
  $mdl  = $v['model'] ?? ($v['vehicle']["model"] ?? null);
  $var  = $v['variant'] ?? ($v['vehicle']["variant"] ?? null);
  $desc = $v['description'] ?? ($v['details']["description"] ?? '');
  $mil  = $v['mileageInKm'] ?? $v['mileageKm'] ?? $v['mileage'] ?? null;
  $body = $v['bodyType'] ?? ($v['vehicle']["bodyType"] ?? null);
  $fuel = $v['fuelType'] ?? ($v['vehicle']["fuelType"] ?? null);
  $trans= $v['transmission'] ?? ($v['vehicle']["transmission"] ?? null);
  $col  = $v['colour'] ?? $v['color'] ?? null;
  $loc  = $v['dealerContactInformation']['city'] ?? $v['dealer']['city'] ?? $v['location'] ?? null;
  $stk  = $v['stockNumber'] ?? $v['stockNo'] ?? null;
  $imgs = [];
  if (!empty($v['imageUrls'])) $imgs = $v['imageUrls'];
  if (!$imgs && !empty($v['images'])) {
    foreach ($v['images'] as $im) {
      if (is_string($im)) $imgs[] = $im;
      elseif (!empty($im['url'])) $imgs[] = $im['url'];
    }
  }
  $pRaw = $v['price'] ?? ($v['pricing']['amount'] ?? null);
  if (is_array($pRaw)) $pRaw = $pRaw['amount'] ?? null;
  $price = null;
  if ($pRaw !== null) $price = (float)preg_replace('/[^\d.]/', '', str_replace(',', '.', (string)$pRaw));
  return [
    'id'=>$id,'make'=>$mk,'model'=>$mdl,'variant'=>$var,'year'=>$yr ? (int)$yr:null,
    'mileageInKm'=>$mil ? (int)$mil:null,'price'=>$price,'colour'=>$col,'description'=>$desc ?: '',
    'imageUrls'=>array_values(array_filter($imgs)),'bodyType'=>$body,'fuelType'=>$fuel,'transmission'=>$trans,
    'stockNumber'=>$stk,'registration'=>$v['registrationNumber'] ?? null,'location'=>$loc,
    'updated'=>$v['updated'] ?? ($v['lastModified'] ?? null)
  ];
}

function hash_vehicle($v) {
  $cmp = [
    $v['make'],$v['model'],$v['variant'],$v['year'],$v['mileageInKm'],
    $v['price'],$v['colour'],$v['description'],$v['bodyType'],
    $v['fuelType'],$v['transmission'],$v['stockNumber'],
    implode('|', $v['imageUrls'] ?? [])
  ];
  return sha1(json_encode($cmp));
}

function slugify($text) {
  $text = preg_replace('~[^\pL\d]+~u', '-', $text);
  $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
  $text = preg_replace('~[^-\w]+~', '', $text);
  $text = trim($text, '-');
  $text = preg_replace('~-+~', '-', $text);
  return strtolower($text ?: 'vehicle');
}
?>